-- SQL --  1. 创建unisql模式
CREATE SCHEMA IF NOT EXISTS unisql;
-- SQL --  授权模式给public
GRANT USAGE ON SCHEMA unisql TO PUBLIC;
-- SQL --  授予所有用户对 unisql schema 中所有表的查询权限
GRANT SELECT ON ALL TABLES IN SCHEMA unisql TO PUBLIC;
-- SQL --  设置默认权限，确保未来新建的表、函数、存储过程自动赋予 PUBLIC 相应权限
ALTER DEFAULT PRIVILEGES IN SCHEMA unisql GRANT SELECT ON TABLES TO PUBLIC;

CREATE OR REPLACE FUNCTION unisql.decode_eq(anyelement, anyelement)
    RETURNS boolean
    LANGUAGE sql
    IMMUTABLE
as
    $function$ SELECT CASE WHEN $1 IS NULL THEN $2 IS NULL ELSE $1 = $2 END; $function$;


-- SQL --  2. 创建dba_tables视图
do language 'plpgsql'
$$
declare
create_mark int;
begin
select count(1) into create_mark from information_schema.views where table_schema='unisql' and table_name='dba_tables';
if create_mark = 0 then
CREATE OR REPLACE VIEW unisql.dba_tables AS
SELECT
    case when n.nspname::text = lower(n.nspname::text)
        then UPPER(n.nspname::text) else n.nspname::text end AS OWNER,
    case when c.relname::text = lower(c.relname::text)
        then UPPER(c.relname::text) else c.relname::text end AS TABLE_NAME,
    coalesce(UPPER(t.spcname::text), 'DEFAULT') as TABLESPACE_NAME,
    NULL::text as CLUSTER_NAME,
    NULL::text as IOT_NAME,
    NULL::text as STATUS,
    NULL::numeric as PCT_FREE,
    NULL::numeric as PCT_USED,
    NULL::numeric as INI_TRANS,
    NULL::numeric as MAX_TRANS,
    NULL::numeric as INITIAL_EXTENT,
    NULL::numeric as NEXT_EXTENT,
    NULL::numeric as MIN_EXTENTS,
    NULL::numeric as MAX_EXTENTS,
    NULL::numeric as PCT_INCREASE,
    NULL::numeric as FREELISTS,
    NULL::numeric as FREELIST_GROUPS,
    case c.relpersistence
        when 'p' then 'YES' else 'NO' end as LOGGING,
    NULL::text as BACKED_UP,
    c.reltuples::bigint as NUM_ROWS,
    c.relpages::bigint as BLOCKS,
    NULL::numeric as EMPTY_BLOCKS,
    NULL::numeric as AVG_SPACE,
    NULL::numeric as CHAIN_CNT,
    NULL::numeric as AVG_ROW_LEN,
    NULL::numeric as AVG_SPACE_FREELIST_BLOCKS,
    NULL::numeric as NUM_FREELIST_BLOCKS,
    NULL::text as DEGREE,
    NULL::text as INSTANCES,
    NULL::text as CACHE,
    NULL::text as TABLE_LOCK,
    NULL::numeric as SAMPLE_SIZE,
    coalesce(pg_stat_get_last_analyze_time(c.oid), pg_stat_get_last_autoanalyze_time(c.oid)) as LAST_ANALYZED,
    case when c.relkind = 'p' then 'YES' else 'NO' end as PARTITIONED,
    NULL::text as IOT_TYPE,
    case c.relpersistence
        when 't' then 'YES' else 'NO' end as TEMPORARY,
    NULL::text as SECONDARY,
    NULL::text as NESTED,
    NULL::text as BUFFER_POOL,
    NULL::text as FLASH_CACHE,
    NULL::text as CELL_FLASH_CACHE,
    'ENABLE' as ROW_MOVEMENT,
    NULL::text as GLOBAL_STATS,
    NULL::text as USER_STATS,
    NULL::text as DURATION,
    NULL::text as SKIP_CORRUPT,
    NULL::text as MONITORING,
    NULL::text as CLUSTER_OWNER,
    NULL::text as DEPENDENCIES,
    NULL::text as COMPRESSION,
    NULL::text as COMPRESS_FOR,
    NULL::text as DROPPED,
    NULL::text as READ_ONLY,
    NULL::text as SEGMENT_CREATED,
    NULL::text as RESULT_CACHE
FROM pg_catalog.pg_class c
JOIN pg_catalog.pg_namespace n
    ON c.relnamespace = n.oid
LEFT JOIN pg_catalog.pg_tablespace t
    ON c.reltablespace = t.oid
WHERE c.relkind IN ('r', 'f', 'p')
    AND (pg_partition_root(c.oid) IS NULL OR pg_partition_root(c.oid) = c.oid)
    AND n.nspname::text NOT LIKE 'pg_toast%';
raise notice 'create view %','unisql.dba_tables';
else
raise notice 'do not create view %','unisql.dba_tables';
end if;
end $$;

-- SQL --  3. 创建all_tables视图
do language 'plpgsql'
$$
declare
create_mark int;
begin
select count(1) into create_mark from information_schema.views where table_schema='unisql' and table_name='all_tables';
if create_mark = 0 then
CREATE OR REPLACE VIEW unisql.all_tables AS select * from unisql.dba_tables;
raise notice 'create view %','unisql.all_tables';
else
raise notice 'do not create view %','unisql.all_tables';
end if;
end $$;

-- SQL --  4. 创建months_between函数
CREATE OR REPLACE FUNCTION unisql.months_between( p_ts1 timestamptz
                                                    , p_ts2 timestamptz)
RETURNS double precision IMMUTABLE strict parallel safe AS $$
select case when dd1 = dd2 or (m1 = dd1 and m2 = dd2)
                then yd * 12 + mmd
            else yd * 12 + mmd
                + (dd1 - dd2) / 31
                + hd / 31 / 24
                + md / 31 / 24 / 60
                + sd / 31 / 24 / 60 / 60
           end
from (select extract('day' from date_trunc('month', p_ts1) + interval '1 month -1 day') as m1
           , extract('day' from date_trunc('month', p_ts2) + interval '1 month -1 day') as m2
           , extract('year' from p_ts1) - extract('year' from p_ts2) as yd
           , extract('month' from p_ts1) - extract('month' from p_ts2) as mmd
           , extract('day' from p_ts1) as dd1
           , extract('day' from p_ts2) as dd2
           , extract('hour' from p_ts1) - extract('hour' from p_ts2) as hd
           , extract('minute' from p_ts1) - extract('minute' from p_ts2) as md
           , extract('second' from p_ts1) - extract('second' from p_ts2) as sd
     ) as x;
$$ LANGUAGE sql;

-- SQL --  5. 创建instr函数
CREATE OR REPLACE FUNCTION unisql.instr(string varchar, string_to_search varchar,
                      beg_index integer, occur_index integer)
    RETURNS integer AS $$
DECLARE
pos integer NOT NULL DEFAULT 0;
    occur_number integer NOT NULL DEFAULT 0;
    temp_str varchar;
    beg integer;
    i integer;
    length integer;
    ss_length integer;
begin
	if beg_index = 0 then
		return 0;
    elseIF beg_index > 0 THEN
        beg := beg_index;
        temp_str := substring(string FROM beg_index);
FOR i IN 1..occur_index LOOP
            pos := position(string_to_search IN temp_str);
            IF i = 1 THEN
                beg := beg + pos - 1;
ELSE
                beg := beg + pos;
END IF;
            temp_str := substring(string FROM beg + 1);
END LOOP;
        IF pos = 0 THEN
            RETURN 0;
ELSE
            RETURN beg;
END IF;
ELSE
        ss_length := char_length(string_to_search);
        length := char_length(string);
        beg :=  LEAST (length + beg_index ,length - ss_length) + 1  ;
        raise info '%,%,%',length + beg_index ,length - ss_length,beg;
        WHILE beg > 0 LOOP
            temp_str := substring(string FROM beg FOR ss_length);
            pos := position(string_to_search IN temp_str);
            IF pos > 0 THEN
                occur_number := occur_number + 1;
                IF occur_number = occur_index THEN
                    RETURN beg;
END IF;
END IF;
            beg := beg - 1;
END LOOP;
RETURN 0;
END IF;
END;
$$ LANGUAGE plpgsql STRICT IMMUTABLE;

-- SQL --  6. 创建instr函数
CREATE OR REPLACE FUNCTION unisql.instr(string varchar, string_to_search varchar,
                      beg_index integer)
    RETURNS integer AS $$
declare
occur_index integer NOT NULL DEFAULT 1;
pos integer NOT NULL DEFAULT 0;
    occur_number integer NOT NULL DEFAULT 0;
    temp_str varchar;
    beg integer;
    i integer;
    length integer;
    ss_length integer;

begin
	if beg_index = 0 then
		return 0;
    elseIF beg_index > 0 THEN
        beg := beg_index;
        temp_str := substring(string FROM beg_index);
FOR i IN 1..occur_index LOOP
            pos := position(string_to_search IN temp_str);
            IF i = 1 THEN
                beg := beg + pos - 1;
ELSE
                beg := beg + pos;
END IF;
            temp_str := substring(string FROM beg + 1);
END LOOP;
        IF pos = 0 THEN
            RETURN 0;
ELSE
            RETURN beg;
END IF;
ELSE
        ss_length := char_length(string_to_search);
        length := char_length(string);
        beg :=  LEAST (length + beg_index ,length - ss_length) + 1  ;
        raise info '%,%,%',length + beg_index ,length - ss_length,beg;
        WHILE beg > 0 LOOP
            temp_str := substring(string FROM beg FOR ss_length);
            pos := position(string_to_search IN temp_str);
            IF pos > 0 THEN
                occur_number := occur_number + 1;
                IF occur_number = occur_index THEN
                    RETURN beg;
END IF;
END IF;
            beg := beg - 1;
END LOOP;
RETURN 0;
END IF;
END;
$$ LANGUAGE plpgsql STRICT IMMUTABLE;

-- SQL --  7. 创建regexp_substr函数
CREATE OR REPLACE FUNCTION unisql.regexp_substr(text, text)

RETURNS text

AS $$

DECLARE

v_substr text;

  v_pattern text;

BEGIN

  v_pattern := '(' || $2 || ')';

  v_substr := (SELECT (regexp_matches($1, v_pattern, 'pg'))[1] OFFSET 0 LIMIT 1);

RETURN v_substr;

END;

$$

LANGUAGE plpgsql;

-- SQL --  8. 创建regexp_substr函数
CREATE OR REPLACE FUNCTION unisql.regexp_substr(text, text, int)

RETURNS text

AS $$

DECLARE

v_substr text;

  v_pattern text;

BEGIN

  IF $3 < 1 THEN

    RAISE EXCEPTION 'argument ''position'' must be a number greater than 0';

END IF;

  v_pattern := '(' || $2 || ')';

  v_substr := (SELECT (regexp_matches(substr($1, $3), v_pattern, 'pg'))[1] OFFSET 0 LIMIT 1);

RETURN v_substr;

END;

$$

LANGUAGE plpgsql;

-- SQL --  9. 创建regexp_substr函数
CREATE OR REPLACE FUNCTION unisql.regexp_substr(text, text, integer, integer)

RETURNS text

AS $$

DECLARE

v_substr text;

  v_pattern text;

BEGIN

  IF $3 < 1 THEN

    RAISE EXCEPTION 'argument ''position'' must be a number greater than 0';

END IF;

  IF $4 < 1 THEN

    RAISE EXCEPTION 'argument ''occurence'' must be a number greater than 0';

END IF;

  v_pattern := '(' || $2 || ')';

  v_substr := (SELECT (regexp_matches(substr($1, $3), v_pattern, 'pg'))[1] OFFSET $4 - 1 LIMIT 1);

RETURN v_substr;

END;

$$

LANGUAGE plpgsql;


-- SQL --  10. 创建regexp_substr函数
CREATE OR REPLACE FUNCTION unisql.regexp_substr(text, text, integer, integer, text)
RETURNS text
AS $$
DECLARE
v_substr text;
  v_pattern text;
  modifiers text;
BEGIN
  IF $1 IS NULL OR $2 IS NULL OR $3 IS NULL OR $4 IS NULL THEN
    RETURN NULL;
END IF;
  IF $3 < 1 THEN
    RAISE EXCEPTION 'argument ''position'' must be a number greater than 0';
END IF;
  IF $4 < 1 THEN
    RAISE EXCEPTION 'argument ''occurence'' must be a number greater than 0';
END IF;

  modifiers := unisql.translate_oracle_modifiers($5, true);
  v_pattern := '(' || $2 || ')';
  v_substr := (SELECT (regexp_matches(substr($1, $3), v_pattern, modifiers))[1] OFFSET $4 - 1 LIMIT 1);
RETURN v_substr;
END;
$$
LANGUAGE plpgsql;

-- SQL --  11. 创建regexp_substr函数
CREATE OR REPLACE FUNCTION unisql.regexp_substr(text, text, integer, integer, text, int)
RETURNS text
AS $$
DECLARE
v_substr text;
  v_pattern text;
  modifiers text;
  v_subexpr integer := $6;
  has_group integer;
BEGIN
  IF $1 IS NULL OR $2 IS NULL OR $3 IS NULL OR $4 IS NULL OR $6 IS NULL THEN
    RETURN NULL;
END IF;
  IF $3 < 1 THEN
    RAISE EXCEPTION 'argument ''position'' must be a number greater than 0';
END IF;
  IF $4 < 1 THEN
    RAISE EXCEPTION 'argument ''occurence'' must be a number greater than 0';
END IF;
  IF v_subexpr < 0 THEN
    RAISE EXCEPTION 'argument ''group'' must be a positive number';
END IF;

  has_group := (SELECT count(*) FROM regexp_matches($2, '(?:[^\\]|^)\(', 'g'));
  IF $6 = 1 AND has_group = 0 THEN
    RETURN NULL;
END IF;

  modifiers := unisql.translate_oracle_modifiers($5, true);

  IF v_subexpr = 0 THEN
    v_pattern := '(' || $2 || ')';
    v_subexpr := 1;
ELSE
    v_pattern := $2;
END IF;

  v_substr := (SELECT (regexp_matches(substr($1, $3), v_pattern, modifiers))[v_subexpr] OFFSET $4 - 1 LIMIT 1);
RETURN v_substr;
END;
$$
LANGUAGE plpgsql;

-- SQL --  12. 创建last_day函数
CREATE OR REPLACE FUNCTION unisql.last_day(TIMESTAMPTZ)
RETURNS TIMESTAMP
AS $$ SELECT (date_trunc('MONTH', $1) + INTERVAL '1 MONTH - 1 day' + $1::time)::timestamp(0); $$
LANGUAGE SQL IMMUTABLE STRICT;

-- SQL --  13. 创建translate_oracle_modifiers函数
CREATE OR REPLACE FUNCTION unisql.translate_oracle_modifiers(text, bool)
RETURNS text
AS $$
DECLARE
modifiers text;
BEGIN
  IF $1 ~ '[^icnsmx]' THEN
    RAISE EXCEPTION 'argument ''flags'' has unsupported modifier(s).';
END IF;
  modifiers := translate($1, 'nm', 'sn');
  IF $2 THEN
    modifiers := modifiers || 'g';
END IF;
RETURN modifiers;
END;
$$
LANGUAGE plpgsql;

-- SQL --  14. 创建regexp_like函数
CREATE OR REPLACE FUNCTION unisql.regexp_like(text, text)
 RETURNS boolean
 LANGUAGE sql
AS $function$
SELECT CASE WHEN (count(*) > 0) THEN true ELSE false END FROM regexp_matches($1, $2, 'p');
$function$;

-- SQL --  15. 创建regexp_like函数
CREATE OR REPLACE FUNCTION unisql.regexp_like(text, text, text)
 RETURNS boolean
 LANGUAGE plpgsql
AS $function$
DECLARE
modifiers text;
BEGIN
  modifiers := unisql.translate_oracle_modifiers($3, false);
  IF (regexp_matches($1, $2, modifiers))[1] IS NOT NULL THEN
    RETURN true;
END IF;
RETURN false;
END;
$function$;

-- SQL --  16. 创建regexp_instr自定义函数，REGEXP_INSTR( string text, pattern text ) -> integer
CREATE OR REPLACE FUNCTION unisql.regexp_instr(text, text)
RETURNS integer
AS $$
DECLARE
v_pos integer;
  v_pattern text;
BEGIN
  IF $1 IS NULL OR $1 = '' OR $2 IS NULL OR $2 = '' THEN
    RETURN NULL;
  END IF;
  v_pattern := '(' || $2 || ')';
  v_pos := (SELECT position((SELECT (regexp_matches($1, v_pattern, 'pg'))[1] OFFSET 0 LIMIT 1) IN $1));
  IF v_pos IS NOT NULL THEN
    RETURN v_pos;
END IF;
RETURN 0;
END;
$$
LANGUAGE plpgsql;

-- SQL --  17. 创建regexp_instr自定义函数，REGEXP_INSTR( string text, pattern text, position int ) -> integer
CREATE OR REPLACE FUNCTION unisql.regexp_instr(text, text, integer)
RETURNS integer
AS $$
DECLARE
v_pos integer;
  v_pattern text;
BEGIN
  IF $1 IS NULL OR $1 = '' OR $2 IS NULL OR $2 = '' OR $3 IS NULL THEN
    RETURN NULL;
  END IF;
  IF $3 < 1 THEN
    RAISE EXCEPTION 'argument ''position'' must be a number greater than 0';
  END IF;
  v_pattern := '(' || $2 || ')';
SELECT position((SELECT (regexp_matches(substr($1, $3), v_pattern, 'pg'))[1] OFFSET 0 LIMIT 1) IN substr($1, $3))
INTO v_pos;
IF v_pos IS NOT NULL THEN
    RETURN v_pos + $3 - 1;
END IF;
RETURN 0;
END;
$$
LANGUAGE plpgsql;

-- SQL --  18. 创建regexp_instr自定义函数，REGEXP_INSTR( string text, pattern text, position int, occurence int ) -> integer
CREATE OR REPLACE FUNCTION unisql.regexp_instr(
    subject TEXT,
    pattern TEXT,
    start_position INTEGER ,
    nth_occurrence INTEGER
)
RETURNS INTEGER AS $$
DECLARE
parts TEXT[];
    match_pos INTEGER := 0;
    i INTEGER := 1;
    match_record RECORD;
    match_record_array TEXT[]:= '{}';
    match_cursor CURSOR FOR
SELECT (regexp_matches(SUBSTRING(subject,start_position), pattern, 'pg'))[1];
BEGIN
    IF subject IS NULL OR subject = '' OR
       pattern IS NULL OR pattern = '' OR
       start_position IS NULL  OR
       nth_occurrence IS NULL
    THEN
        RETURN NULL;
    END IF;
    IF start_position < 1 THEN
      RAISE EXCEPTION 'argument ''position'' must be a number greater than 0';
    END IF;
    IF nth_occurrence < 1 THEN
      RAISE EXCEPTION 'argument ''occurence'' must be a number greater than 0';
    END IF;
    IF start_position > LENGTH(subject) THEN
        RETURN 0;
    END IF;
    parts := regexp_split_to_array(SUBSTRING(subject,start_position), pattern);
    IF array_length(parts, 1) <= nth_occurrence THEN
        RETURN 0;
    END IF;
OPEN match_cursor;
LOOP
FETCH match_cursor INTO match_record;
        EXIT WHEN NOT FOUND;
        match_record_array := array_append(match_record_array, match_record.regexp_matches);
END LOOP;
CLOSE match_cursor;
FOR i IN 1..nth_occurrence loop
        match_pos := match_pos + LENGTH(parts[i]);
        IF i < nth_occurrence then
          match_pos := match_pos + LENGTH(match_record_array[i]);
        END IF;
END LOOP;
RETURN match_pos + start_position;
END;
$$ LANGUAGE plpgsql;

-- SQL --  19. 创建regexp_instr自定义函数，REGEXP_INSTR( string text, pattern text, position int, occurence int, return_opt int ) -> integer
CREATE OR REPLACE FUNCTION unisql.regexp_instr(
    subject TEXT,
    pattern TEXT,
    start_position INTEGER,
    nth_occurrence INTEGER ,
    return_option INTEGER
)
RETURNS INTEGER AS $$
DECLARE
parts TEXT[];
    match_pos INTEGER := 0;
    i INTEGER := 1;
    match_record RECORD;
    match_record_array TEXT[]:= '{}';
    match_cursor CURSOR FOR
SELECT (regexp_matches(SUBSTRING(subject,start_position), pattern, 'pg'))[1];
BEGIN
    IF subject IS NULL OR subject = '' OR
       pattern IS NULL OR pattern = '' OR
       start_position IS NULL  OR
       nth_occurrence IS NULL  OR
       return_option IS NULL
    THEN
        RETURN NULL;
    END IF;
    IF start_position < 1 THEN
      RAISE EXCEPTION 'argument ''position'' must be a number greater than 0';
    END IF;
    IF nth_occurrence < 1 THEN
      RAISE EXCEPTION 'argument ''occurence'' must be a number greater than 0';
    END IF;
    IF return_option != 0 AND return_option != 1 THEN
      RAISE EXCEPTION 'argument ''return_opt'' must be 0 or 1';
    END IF;
    IF start_position > LENGTH(subject) THEN
        RETURN 0;
    END IF;
    parts := regexp_split_to_array(SUBSTRING(subject,start_position), pattern);
    IF array_length(parts, 1) <= nth_occurrence THEN
        RETURN 0;
    END IF;
OPEN match_cursor;
LOOP
FETCH match_cursor INTO match_record;
	        EXIT WHEN NOT FOUND;
	        match_record_array := array_append(match_record_array, match_record.regexp_matches);
END LOOP;
CLOSE match_cursor;
FOR i IN 1..nth_occurrence loop
        match_pos := match_pos + LENGTH(parts[i]);
        IF (i < nth_occurrence AND return_option = 0 ) OR return_option =1 then
          match_pos := match_pos + LENGTH(match_record_array[i]);
        END IF;
END LOOP;
RETURN match_pos + start_position;
END;
$$ LANGUAGE plpgsql;

-- SQL --  20. 创建substr函数,参数类型分别为text, numeric,numeric
CREATE OR REPLACE FUNCTION unisql.substr(text, numeric,numeric)
RETURNS TEXT LANGUAGE plpgsql IMMUTABLE STRICT AS  $$
DECLARE
processed_text TEXT;
    processed_index INT;
    processed_length INT;
begin
     IF $1 = '' THEN
        RETURN NULL;
END IF;
    processed_text := $1;
    IF TRUNC($2) = 0 THEN
        processed_index := 1;
    ELSIF TRUNC($2) < 0 THEN
        processed_index := LENGTH(processed_text) + TRUNC($2)::INT + 1;
        IF processed_index < 1 THEN
            RETURN NULL;
END IF;
ELSE
        processed_index := TRUNC($2)::INT;
END IF;
   processed_length := TRUNC($3);
   IF processed_length < 1 then
        return null;
END IF;
RETURN pg_catalog.substr(processed_text, processed_index,processed_length);
END;
$$;

-- SQL --  21. 创建substr函数,参数类型分别为text, numeric
CREATE OR REPLACE FUNCTION unisql.substr(text, numeric)
RETURNS TEXT LANGUAGE plpgsql IMMUTABLE STRICT AS  $$
DECLARE
processed_text TEXT;
    processed_index INT;
begin
    IF $1 = '' THEN
        RETURN NULL;
END IF;
    processed_text := $1;
    IF TRUNC($2) = 0 THEN
        processed_index := 1;
    ELSIF TRUNC($2) < 0 THEN
        processed_index := LENGTH(processed_text) + TRUNC($2)::INT + 1;
        IF processed_index < 1 THEN
            RETURN NULL;
END IF;
ELSE
        processed_index := TRUNC($2)::INT;
END IF;
RETURN pg_catalog.substr(processed_text, processed_index);
END;
$$;

-- SQL --  22. 创建substring_index函数
CREATE OR REPLACE FUNCTION unisql.substring_index(f_string varchar, f_delimiter varchar, f_position numeric)
RETURNS varchar AS $$
DECLARE
    o_string varchar;
    v_instr_pos int := 0;
    v_start_pos int := 1;
    sub_string varchar;
    sub_position int;
    sub_length int;
    v_position_int int;
BEGIN
    IF f_string IS NULL OR f_string = '' THEN
        RETURN NULL;
    END IF;

    IF f_delimiter IS NULL OR f_delimiter = '' THEN
        RETURN NULL;
    END IF;

    IF f_position IS NULL OR f_position = 0 THEN
        RETURN NULL;
    END IF;

    v_position_int := trunc(f_position);
    
    sub_length := length(f_string) + 1;

    IF v_position_int > 0 THEN
        sub_string := f_string;
        sub_position := v_position_int;
    ELSE
        sub_string := reverse(f_string);
        sub_position := abs(v_position_int);
    END IF;

    FOR i IN 1..sub_position LOOP
        v_instr_pos := strpos(substring(sub_string FROM v_start_pos), f_delimiter);
        EXIT WHEN v_instr_pos = 0; 
        v_start_pos := v_start_pos + v_instr_pos;
    END LOOP;

    IF v_instr_pos = 0 THEN
        RETURN f_string;
    END IF;
    
    IF v_position_int > 0 THEN
        o_string := substring(f_string FROM 1 FOR v_start_pos - 2);
    ELSE
        o_string := substring(f_string FROM sub_length - v_start_pos + 2);
    END IF;

    RETURN o_string;
END;
$$ LANGUAGE plpgsql;

-- SQL --  23. 创建emptystr2null
CREATE OR REPLACE FUNCTION unisql.emptystr2null(var anyelement)
RETURNS anyelement
strict immutable
AS $$ 
DECLARE
	input_text text;
BEGIN
	input_text := var::text;
	if input_text = '' then 
		return null;
	end if;
	return var;
EXCEPTION
	WHEN others THEN
	RETURN var;
end;$$ LANGUAGE plpgsql;

