-- SQL --  创建 unisql 用户并赋权
DECLARE
    v_count NUMBER;
BEGIN
    SELECT COUNT(*) INTO v_count FROM DBA_USERS WHERE USERNAME = 'UNISQL';
    IF v_count = 0 THEN
        EXECUTE IMMEDIATE 'CREATE USER UNISQL IDENTIFIED BY "unisql_pwd123"';
        EXECUTE IMMEDIATE 'GRANT SELECT, INSERT, UPDATE, DELETE ON unisql.* TO PUBLIC';
    END IF;
END;


-- SQL --  通过表名获取该表上一次的自增 id, 类比 mysql, 该表必须有一个字段有 AUTO_INCREMENT 属性, 允许传空字符串
CREATE OR REPLACE FUNCTION unisql.last_insert_id(
    table_name IN VARCHAR2
) RETURN NUMBER IS
    seq_name VARCHAR2(100);
    like_pattern VARCHAR2(100);
    curr_value NUMBER;
BEGIN
    SELECT 'ISEQ$$_' || object_id || '_%' INTO like_pattern
    FROM user_objects
    WHERE object_name = table_name
      AND object_type = 'TABLE';

    SELECT sequence_name INTO seq_name
    FROM user_sequences
    WHERE sequence_name LIKE like_pattern;

    EXECUTE IMMEDIATE 'SELECT ' || seq_name || '.currval FROM dual'
    INTO curr_value;

    RETURN curr_value;
EXCEPTION
	WHEN OTHERS THEN
		RETURN 0;
END;


-- SQL --  创建str_to_timestamp函数，用于模拟MYSQL中 字符串，数值到日期类型的隐式转换
CREATE OR REPLACE FUNCTION unisql.str_to_timestamp(p_input_str VARCHAR2) RETURN TIMESTAMP AS
    v_clean_str    VARCHAR2(30);
    v_year         NUMBER;
    v_length       NUMBER;
   	v_may_month    VARCHAR2(10);
   	v_default_flag BOOL;
BEGIN  
	v_may_month := upper(substr(p_input_str, 4, 3));
	CASE v_may_month
        WHEN 'JAN' THEN v_default_flag := TRUE;
        WHEN 'FEB' THEN v_default_flag := TRUE;
        WHEN 'MAR' THEN v_default_flag := TRUE;
        WHEN 'APR' THEN v_default_flag := TRUE;
        WHEN 'MAY' THEN v_default_flag := TRUE;
        WHEN 'JUN' THEN v_default_flag := TRUE;
        WHEN 'JUL' THEN v_default_flag := TRUE;
        WHEN 'AUG' THEN v_default_flag := TRUE;
        WHEN 'SEP' THEN v_default_flag := TRUE;
        WHEN 'OCT' THEN v_default_flag := TRUE;
        WHEN 'NOV' THEN v_default_flag := TRUE;
        WHEN 'DEC' THEN v_default_flag := TRUE;
        ELSE v_default_flag := FALSE;
    END CASE;
   	
   	IF v_default_flag THEN
   		v_length := LENGTH(p_input_str);
		IF v_length < 9  THEN
			RAISE_APPLICATION_ERROR(-20000, 'Input too short to be a valid date');
		END IF;
	
		IF v_length = 9 THEN
			RETURN TO_TIMESTAMP(p_input_str, 'DD-MON-RR');
		END IF;
	
	    IF INSTR(p_input_str, ' +') > 0 THEN
	    	RETURN TO_TIMESTAMP_TZ(p_input_str, 'DD-MON-RR HH.MI.SSXFF AM TZR');
	    ELSIF INSTR(p_input_str, ' -') > 0 THEN
	      	RETURN TO_TIMESTAMP_TZ(p_input_str, 'DD-MON-RR HH.MI.SSXFF AM TZR'); 
	    ELSE 
	    	RETURN TO_TIMESTAMP(p_input_str, 'DD-MON-RR HH.MI.SSXFF AM');
	    END IF;
   	END IF;
   
    IF INSTR(p_input_str, '. +') > 0 THEN   
    	RETURN TO_TIMESTAMP_TZ(p_input_str, 'YYYY-MM-DD HH24:MI:SS. TZR TZD');
    ELSIF INSTR(p_input_str, '. -') > 0 THEN 
      	RETURN TO_TIMESTAMP_TZ(p_input_str, 'YYYY-MM-DD HH24:MI:SS. TZR TZD');  
    ELSIF INSTR(p_input_str, ' +') > 0 THEN
    	RETURN TO_TIMESTAMP_TZ(p_input_str, 'YYYY-MM-DD HH24:MI:SS.FF TZR TZD');
    ELSIF INSTR(p_input_str, ' -') > 0 THEN
      	RETURN TO_TIMESTAMP_TZ(p_input_str, 'YYYY-MM-DD HH24:MI:SS.FF TZR TZD'); 
    END IF;

    v_clean_str := REGEXP_REPLACE(p_input_str, '[^0-9]', '');
    v_length := LENGTH(v_clean_str);

    IF v_length < 6 THEN
        RAISE_APPLICATION_ERROR(-20000, 'Input too short to be a valid date');
    END IF;
       
    IF v_length = 6 THEN
        v_year := TO_NUMBER(SUBSTR(v_clean_str, 1, 2));
        IF v_year BETWEEN 0 AND 69 THEN
            RETURN TO_TIMESTAMP('20' || v_clean_str, 'YYYYMMDD');
        ELSE
            RETURN TO_TIMESTAMP('19' || v_clean_str, 'YYYYMMDD');
        END IF;
    ELSIF v_length = 8 THEN
        RETURN TO_TIMESTAMP(v_clean_str, 'YYYYMMDD');
    ELSIF v_length = 12 THEN
        v_year := TO_NUMBER(SUBSTR(v_clean_str, 1, 2));
        IF v_year BETWEEN 0 AND 69 THEN
            RETURN TO_TIMESTAMP('20' || v_clean_str, 'YYYYMMDDHH24MISS');
        ELSE
            RETURN TO_TIMESTAMP('19' || v_clean_str, 'YYYYMMDDHH24MISS');
        END IF;
    ELSIF v_length = 14 THEN
        RETURN TO_TIMESTAMP(v_clean_str, 'YYYYMMDDHH24MISS');
    ELSE 
    	RETURN TO_TIMESTAMP(v_clean_str, 'YYYYMMDDHH24MISSFF');
    END IF;
END;

-- SQL --  支持mariadb的weekofyear函数转换到ob_oracle执行
CREATE OR REPLACE FUNCTION unisql.weekofyear(
    date_str IN VARCHAR2
) RETURN NUMBER
IS
BEGIN
    IF date_str IS NULL THEN
        RETURN NULL;
END IF;
RETURN TO_CHAR(unisql.str_to_timestamp(date_str),'IW');
END;

-- SQL --  oceanbase_oracle 中没有 date_add函数，使用自定义函数实现
CREATE OR REPLACE FUNCTION unisql.date_add(
    date_str      IN VARCHAR2,
    interval_val  IN NUMBER,
    interval_unit IN VARCHAR2
) RETURN TIMESTAMP IS
	v_timestamp_tz TIMESTAMP WITH TIME ZONE;
    v_ts TIMESTAMP;
    v_new_date TIMESTAMP;
    v_original_day NUMBER;
    v_last_day TIMESTAMP;
   	v_time_part INTERVAL DAY(9) TO SECOND;
    interval_val_month NUMBER;
BEGIN
    IF date_str IS NULL THEN
        RETURN NULL;
    END IF;

    BEGIN
        v_ts := unisql.str_to_timestamp(date_str);
    EXCEPTION WHEN OTHERS THEN
        RETURN NULL;
    END;

    CASE UPPER(interval_unit)
        WHEN 'SECOND' THEN
                RETURN v_ts + NUMTODSINTERVAL(interval_val, 'SECOND');
        WHEN 'MINUTE' THEN
                RETURN v_ts + NUMTODSINTERVAL(interval_val, 'MINUTE');
        WHEN 'HOUR' THEN
                RETURN v_ts + NUMTODSINTERVAL(interval_val, 'HOUR');
        WHEN 'DAY' THEN
                RETURN v_ts + NUMTODSINTERVAL(interval_val, 'DAY');
        WHEN 'MONTH' THEN
                interval_val_month := interval_val;
                v_new_date := ADD_MONTHS(v_ts, interval_val_month);
                v_original_day := EXTRACT(DAY FROM v_ts);
                v_last_day := LAST_DAY(v_new_date);
                v_time_part := v_ts - TRUNC(v_ts);

                IF v_original_day <= EXTRACT(DAY FROM v_last_day) THEN

                    v_new_date :=
                        TO_TIMESTAMP(
                            TO_CHAR(EXTRACT(YEAR FROM v_new_date), 'FM9999') || '-' ||
                            TO_CHAR(EXTRACT(MONTH FROM v_new_date), 'FM09') || '-' ||
                            TO_CHAR(v_original_day, 'FM09'),
                            'YYYY-MM-DD'
                        ) + v_time_part;
                ELSE
                    v_new_date := TO_TIMESTAMP(TO_CHAR(v_new_date, 'YYYY-MM-DD'), 'YYYY-MM-DD') + v_time_part;
                END IF;
                RETURN v_new_date;

        WHEN 'YEAR' THEN
                interval_val_month := 12 * interval_val;
                v_new_date := ADD_MONTHS(v_ts, interval_val_month);
                v_original_day := EXTRACT(DAY FROM v_ts);
                v_last_day := LAST_DAY(v_new_date);
                v_time_part := v_ts - TRUNC(v_ts);

                IF v_original_day <= EXTRACT(DAY FROM v_last_day) THEN

                    v_new_date :=
                        TO_TIMESTAMP(
                            TO_CHAR(EXTRACT(YEAR FROM v_new_date), 'FM9999') || '-' ||
                            TO_CHAR(EXTRACT(MONTH FROM v_new_date), 'FM09') || '-' ||
                            TO_CHAR(v_original_day, 'FM09'),
                            'YYYY-MM-DD'
                        ) + v_time_part;
                ELSE
                    v_new_date :=
                        TO_TIMESTAMP(TO_CHAR(v_new_date, 'YYYY-MM-DD'), 'YYYY-MM-DD') + v_time_part;
                END IF;
                RETURN v_new_date;
        ELSE
            RAISE_APPLICATION_ERROR(-20002, 'Unsupported interval unit: ' || interval_unit);
    END CASE;
END;

-- SQL --  oceanbase_oracle 中没有 date_sub函数，使用自定义函数实现
CREATE OR REPLACE FUNCTION unisql.date_sub(
    date_str      IN VARCHAR2,
    interval_val  IN NUMBER,
    interval_unit IN VARCHAR2
) RETURN TIMESTAMP IS
    v_new_date TIMESTAMP;
BEGIN
	v_new_date := unisql.date_add(date_str,-interval_val,interval_unit);
    RETURN v_new_date;
END;

-- SQL --  oceanbase_oracle 中没有UUID函数, 使用自定义函数实现
CREATE OR REPLACE FUNCTION unisql.uuid RETURN VARCHAR2
IS
    v_guid RAW(16);
    v_result VARCHAR2(36);
BEGIN
    v_guid := SYS_GUID();
    
    v_result := 
        LOWER(SUBSTR(RAWTOHEX(v_guid), 1, 8)) || '-' ||
        LOWER(SUBSTR(RAWTOHEX(v_guid), 9, 4)) || '-' ||
        LOWER(SUBSTR(RAWTOHEX(v_guid), 13, 4)) || '-' ||
        LOWER(SUBSTR(RAWTOHEX(v_guid), 17, 4)) || '-' ||
        LOWER(SUBSTR(RAWTOHEX(v_guid), 21, 12));
    
    RETURN v_result;
END;

-- SQL --  在赋权前就需要创建unisql_version版本号信息维护表，以便能够被赋权
DECLARE
v_count NUMBER;
BEGIN
SELECT COUNT(*) INTO v_count from ALL_TABLES WHERE OWNER = upper('UNISQL') AND TABLE_NAME = 'UNISQL_VERSION';

IF v_count = 0 THEN
        EXECUTE IMMEDIATE 'create table UNISQL.UNISQL_VERSION (id int primary key,version varchar(20),update_time date)';
END IF;
END;

-- SQL --  创建所有用户对unisql下表访问执行权限
BEGIN
FOR obj IN (
    SELECT object_name
    FROM all_objects
    WHERE owner = upper('UNISQL')
      AND object_type IN ('TABLE')
  ) LOOP
    EXECUTE IMMEDIATE 'GRANT SELECT, INSERT, UPDATE, DELETE ON UNISQL.' || obj.object_name || ' TO PUBLIC';
END LOOP;
END;

-- SQL --  创建所有用户对unisql下视图访问权限
BEGIN
FOR obj IN (
    SELECT object_name
    FROM all_objects
    WHERE owner = upper('UNISQL')
      AND object_type IN ('VIEW')
  ) LOOP
    EXECUTE IMMEDIATE 'GRANT SELECT ON UNISQL.' || obj.object_name || ' TO PUBLIC';
END LOOP;
END;

-- SQL --  创建所有用户对unisql下PROCEDURE、FUNCTION、PACKAGE访问执行权限
BEGIN
FOR obj IN (
    SELECT object_name
    FROM all_objects
    WHERE owner = upper('UNISQL')
      AND object_type IN ('PROCEDURE', 'FUNCTION', 'PACKAGE')
  ) LOOP
    EXECUTE IMMEDIATE 'GRANT EXECUTE ON UNISQL.' || obj.object_name || ' TO PUBLIC';
END LOOP;
END;
