-- SQL --  创建unisql 数据库
CREATE DATABASE IF NOT EXISTS unisql default charset utf8mb4;

-- SQL --  删除GrantUnisqlPermissions
DROP PROCEDURE IF EXISTS unisql.GrantUnisqlPermissions;

-- SQL --  创建GrantUnisqlPermissions
CREATE PROCEDURE unisql.GrantUnisqlPermissions()
BEGIN
  DECLARE done INT DEFAULT 0;
  DECLARE username VARCHAR(100);
  DECLARE cur CURSOR FOR SELECT concat('\'',User, '\'','@','\'',Host,'\'')  FROM mysql.user;
  DECLARE CONTINUE HANDLER FOR NOT FOUND SET done = 1;

  OPEN cur;
  read_loop: LOOP
    FETCH cur INTO username;
    IF done THEN
      LEAVE read_loop;
    END IF;

    SET @grant_stmt = CONCAT('GRANT EXECUTE, SELECT, UPDATE, INSERT ON unisql.* TO ', username, ' with grant option;');
    PREPARE stmt FROM @grant_stmt;
    EXECUTE stmt;
    DEALLOCATE PREPARE stmt;
  END LOOP;

  CLOSE cur;
END ;

-- SQL --  执行GrantUnisqlPermissions
call unisql.GrantUnisqlPermissions();

-- SQL --  创建UNISQL_TRUNC函数
drop function if exists unisql.UNISQL_TRUNC;

-- SQL --  创建UNISQL_TRUNC函数
CREATE FUNCTION unisql.UNISQL_TRUNC(date_value DATETIME, fmt VARCHAR(20))
RETURNS DATETIME
DETERMINISTIC
NO SQL
BEGIN
    CASE
        WHEN fmt IN ('CC', 'SCC') THEN
            RETURN DATE_FORMAT(DATE(date_value), CONCAT((truncate((year(date_value) - 1) / 100, 0) * 100 + 1), '-01-01'));
        WHEN fmt IN ('SYYYY', 'YYYY', 'YEAR', 'SYEAR', 'YYY', 'YY', 'Y') THEN
            RETURN DATE_FORMAT(DATE(date_value), '%Y-01-01');
        WHEN fmt IN ('IYYY', 'IY', 'I') then
            RETURN date_sub(date_value, interval ((weekofyear(date_value) - 1) * 7 + weekday(date_value)) day);
        WHEN fmt = 'Q' THEN
            RETURN DATE_FORMAT(DATE(date_value), CONCAT(DATE_FORMAT(DATE(date_value), '%Y'), '-', QUARTER(DATE(date_value)) * 3 - 2, '-01'));
        WHEN fmt IN ('MONTH', 'MON', 'MM', 'RM') THEN
            RETURN DATE_FORMAT(DATE(date_value), '%Y-%m-01');
        WHEN fmt = 'WW' THEN
            RETURN DATE_FORMAT(DATE_SUB(date_value, INTERVAL (dayofyear(date_value) - 1) % 7 DAY), '%Y-%m-%d 00:00:00');
        WHEN fmt = 'IW' THEN
            RETURN DATE_SUB(date_value, interval weekday(date_value) DAY);
        WHEN fmt = 'W' THEN
            RETURN DATE_FORMAT(DATE_SUB(date_value, INTERVAL (dayofmonth(date_value) - 1) % 7 DAY), '%Y-%m-%d 00:00:00');
        WHEN fmt IN ('DDD', 'DD', 'J') THEN
            RETURN DATE(date_value);
        WHEN fmt IN ('DAY', 'DY', 'D') THEN
            RETURN DATE_FORMAT(DATE_SUB(date_value, interval (dayofweek(date_value) - 1) DAY), '%Y-%m-%d 00:00:00');
        WHEN fmt IN ('HH', 'HH12', 'HH24') THEN
            RETURN DATE_FORMAT(date_value, '%Y-%m-%d %H:00:00');
        WHEN fmt = 'MI' THEN
            RETURN DATE_FORMAT(date_value, '%Y-%m-%d %H:%i:00');
        ELSE
            RETURN NULL;
    END CASE;
END ;

-- SQL --  删除UNISQL_ROUND函数
drop function if exists unisql.UNISQL_ROUND;

-- SQL --  创建UNISQL_ROUND函数，无法支持 IYYY。
CREATE FUNCTION unisql.UNISQL_ROUND(date_value DATETIME, fmt VARCHAR(20))
RETURNS DATETIME
DETERMINISTIC
NO SQL
BEGIN
    CASE
        WHEN fmt IN ('CC', 'SCC') THEN
            RETURN CONCAT((YEAR(date_value) DIV 100 * 100) + 1, '-01-01 00:00:00');
        WHEN fmt IN ('SYYYY', 'YYYY', 'YEAR', 'SYEAR', 'YYY', 'YY', 'Y') THEN
            RETURN CONCAT(IF(MONTH(date_value) >= 7, DATE_FORMAT(DATE(date_value), '%Y') + 1, DATE_FORMAT(DATE(date_value), '%Y')), '-01-01 00:00:00');
        WHEN fmt IN ('IYYY', 'IY', 'I') then
        begin

               declare lo datetime;
               declare hi datetime;
               declare d datetime;
               declare iso_year int;
            set lo = date_sub(date_value, interval ((weekofyear(date_value) - 1) * 7 + weekday(date_value)) day);
            set hi = date_add(lo, interval (GREATEST(weekofyear(concat(year(lo), '-12-31')), weekofyear(concat(year(lo), '-12-26'))) * 7) day);
            if month(date_value) = 1 and weekofyear(date_value) >= 50 then
                set iso_year = year(date_value) - 1;
            elseif month(date_value) = 12 and weekofyear(date_value) = 1 then
                set iso_year = year(date_value) + 1;
            else
                set iso_year = year(date_value);
            end if;
            set d = concat(iso_year, date_format(date_value, '-%m-%d'));
            if datediff(d, lo) < datediff(hi, d) then
                return lo;
            else
                return hi;
            end if;
        end;
        WHEN fmt = 'Q' THEN
          BEGIN
            DECLARE year INT;
            DECLARE month INT;
            DECLARE day INT;
            DECLARE quarter_start_month INT;
            SET year = YEAR(date_value);
            SET month = MONTH(date_value);
            SET day = DAY(date_value);
            IF month = 1 OR (month = 2 AND day <= 15) THEN
              SET quarter_start_month = 1;
            ELSEIF (month = 2 AND day >= 16) OR month <= 4 or (month <= 5 AND day <= 15) THEN
              SET quarter_start_month = 4;
            ELSEIF (month = 5 AND day >= 16) OR month <= 7 or (month <= 8 AND day <= 15) THEN
              SET quarter_start_month = 7;
            ELSEIF (month = 8 AND day >= 16) OR month <= 10 or (month <= 11 AND day <= 15) THEN
              SET quarter_start_month = 10;
            ELSE
              SET year = year + 1;
              SET quarter_start_month = 1;
            END IF;
            return STR_TO_DATE(CONCAT(year, '-', LPAD(quarter_start_month, 2, '00'), '-01 00:00:00'), '%Y-%m-%d %H:%i:%s');
          END;
        WHEN fmt IN ('MONTH', 'MON', 'MM', 'RM') THEN
            RETURN DATE_FORMAT(IF(DAY(date_value) >= 16, DATE_FORMAT(DATE_ADD(DATE(date_value), INTERVAL 1 MONTH), '%Y-%m-01'), DATE_FORMAT(DATE(date_value), '%Y-%m-01')), '%Y-%m-%d 00:00:00');
        WHEN fmt = 'WW' then
            begin
	            declare lo datetime;
	            declare hi datetime;
	            set lo = date_format(date_sub(date_value, interval (dayofyear(date_value) - 1) % 7 day), '%Y-%m-%d 00:00:00');
	            set hi = date_add(lo, interval 7 day);
	            if timestampdiff(second, lo, date_value) < timestampdiff(second, date_value, hi) then
	                return lo;
	            else
	                return hi;
	            end if;
            end;
        WHEN fmt = 'IW' THEN
        begin
            declare lo datetime;
            declare hi datetime;
            set lo = date_format(date_sub(date_value, interval weekday(date_value) day), '%Y-%m-%d 00:00:00');
            set hi = date_add(lo, interval 7 day);
            if timestampdiff(second, lo, date_value) < timestampdiff(second, date_value, hi) then
                return lo;
            else
                return hi;
            end if;
        end;
        WHEN fmt = 'W' THEN
        begin
            declare lo datetime;
            declare hi datetime;
            set lo = date_format(DATE_SUB(date_value, INTERVAL (dayofmonth(date_value) - 1) % 7 DAY), '%Y-%m-%d 00:00:00');
            set hi = date_add(lo, interval 7 day);
            if timestampdiff(second, lo, date_value) < timestampdiff(second, date_value, hi) then
                return lo;
            else
                return hi;
            end if;
        end;
        WHEN fmt IN ('DDD', 'DD', 'J') then
        begin
            declare lo datetime;
            declare hi datetime;
            set lo = DATE(date_value);
            set hi = date_add(lo, interval 1 day);
            if timestampdiff(second, lo, date_value) < timestampdiff(second, date_value, hi) then
                return lo;
            else
                return hi;
            end if;
        end;
        WHEN fmt IN ('DAY', 'DY', 'D') THEN
        begin
            declare lo datetime;
            declare hi datetime;
            set lo = date_format(DATE_SUB(DATE(date_value), interval (dayofweek(date_value) - 1) DAY), '%Y-%m-%d 00:00:00');
            set hi = date_add(lo, interval 7 day);
            if timestampdiff(second, lo, date_value) < timestampdiff(second, date_value, hi) then
                return lo;
            else
                return hi;
            end if;
        end;
        WHEN fmt IN ('HH', 'HH12', 'HH24') THEN
        begin
            declare lo datetime;
            declare hi datetime;
            set lo = DATE_FORMAT(date_value, '%Y-%m-%d %H:00:00');
            set hi = date_add(lo, interval 1 hour);
            if timestampdiff(second, lo, date_value) < timestampdiff(second, date_value, hi) then
                return lo;
            else
                return hi;
            end if;
        end;
        WHEN fmt = 'MI' THEN
        begin
            declare lo datetime;
            declare hi datetime;
            set lo = DATE_FORMAT(date_value, '%Y-%m-%d %H:%i:00');
            set hi = date_add(lo, interval 1 minute);
            if timestampdiff(second, lo, date_value) < timestampdiff(second, date_value, hi) then
                return lo;
            else
                return hi;
            end if;
        end;
        ELSE
            RETURN NULL;
    END CASE;
END ;

-- SQL --  删除UNISQL_LTRIM函数
drop function if exists unisql.UNISQL_LTRIM;

-- SQL --  创建UNISQL_LTRIM函数
CREATE FUNCTION unisql.UNISQL_LTRIM(input_str text, chars_to_remove VARCHAR(255))
RETURNS VARCHAR(255)
DETERMINISTIC
NO SQL
begin
    IF input_str IS NULL or chars_to_remove is null or length(chars_to_remove) = 0 THEN
        RETURN NULL;
    END IF;

    SET @c = input_str;
    WHILE length(@c) > 0 and LOCATE(LEFT(@c, 1), chars_to_remove) != 0 DO
        SET @c = substring(@c, 2);
    END WHILE;
    RETURN @c;
END ;

-- SQL --  删除UNISQL_RTRIM函数
drop function if exists unisql.UNISQL_RTRIM;

-- SQL --  创建UNISQL_RTRIM函数
CREATE FUNCTION unisql.UNISQL_RTRIM(input_str text, chars_to_remove VARCHAR(255))
RETURNS VARCHAR(255)
DETERMINISTIC
NO SQL
begin
    IF input_str IS NULL or chars_to_remove is null or length(chars_to_remove) = 0 THEN
        RETURN NULL;
    END IF;

    SET @c = input_str;
    set @l = length(@c);
    WHILE @l > 0 and LOCATE(RIGHT(@c, 1), chars_to_remove) != 0 DO
        SET @c = substring(@c, 1, @l - 1);
        set @l = @l - 1;
    END WHILE;
    RETURN @c;
END ;

-- SQL --  创建序列表
CREATE TABLE if not exists unisql.`UNISQL_SEQUENCE` (
  `SEQ_CODE` VARCHAR(64) NOT NULL COMMENT '序列代码',
  `SEQ_NAME` varchar(256) DEFAULT NULL COMMENT '序列名称',
  `CURRENT_VAL` BIGINT(16) NOT NULL COMMENT '当前序列值',
  `INCREMENT_VAL` BIGINT(16) NOT NULL DEFAULT '1' COMMENT '增长值',
  PRIMARY KEY (`SEQ_CODE`)
) ENGINE=INNODB DEFAULT CHARSET=utf8 COMMENT='序列表';

-- SQL --  如果存在序列函数则删除
DROP FUNCTION IF EXISTS unisql.`CURRVAL`;

-- SQL --  创建序列函数
CREATE FUNCTION unisql.`CURRVAL` (V_SEQ_CODE VARCHAR (64))
    RETURNS BIGINT (16)
    READS SQL DATA
BEGIN
  DECLARE VALUE BIGINT;
  SELECT CURRENT_VAL INTO VALUE FROM unisql.`UNISQL_SEQUENCE` WHERE SEQ_CODE = V_SEQ_CODE;
  if VALUE is null then
    SIGNAL SQLSTATE '02000' SET MESSAGE_TEXT = 'Sequence not found';
  end if;
  RETURN VALUE;
END ;

-- SQL --  如果存在nextval函数则删除
DROP FUNCTION IF EXISTS unisql.`NEXTVAL`;

-- SQL --  创建nextval函数
CREATE FUNCTION unisql.`NEXTVAL`(V_SEQ_CODE VARCHAR(64))
    RETURNS BIGINT(16)
    MODIFIES SQL DATA
BEGIN
    UPDATE unisql.UNISQL_SEQUENCE SET CURRENT_VAL = CURRENT_VAL + INCREMENT_VAL WHERE SEQ_CODE = V_SEQ_CODE;
    if ROW_COUNT() = 0 then
        SIGNAL SQLSTATE '02000' SET MESSAGE_TEXT = 'Sequence not found';
    end if;
    RETURN unisql.CURRVAL(V_SEQ_CODE);
END ;


-- SQL --  删除UNISQL_TO_DATE函数。Oracle 在 format 中缺少年月日时，会使用当前年月日填充，这个函数就是模仿这种行为。lacks: 由 YMD 三个字符中的一到多个组成，表示 fmt 中缺少了年月日中哪一部分
DROP FUNCTION IF EXISTS unisql.`UNISQL_TO_DATE`;

-- SQL --  创建UNISQL_TO_DATE函数。Oracle 在 format 中缺少年月日时，会使用当前年月日填充，这个函数就是模仿这种行为。lacks: 由 YMD 三个字符中的一到多个组成，表示 fmt 中缺少了年月日中哪一部分
CREATE FUNCTION unisql.`UNISQL_TO_DATE`(str varchar(64), fmt VARCHAR(64), lacks varchar(16))
    RETURNS datetime(6)
    DETERMINISTIC
    NO SQL
begin
    declare v_fmt varchar(64);
    declare v_str varchar(64);
    declare v_now varchar(64);
    set v_fmt = fmt;
    set v_str = str;
    set v_now = DATE_SUB(sysdate(), interval day(sysdate()) - 1 day);
    if instr(lacks, 'Y') > 0 then
        set v_fmt = concat('%Y-', v_fmt);
        set v_str = concat(date_format(v_now, '%Y'), concat('-', v_str));
    end if;
    if instr(lacks, 'M') > 0 then
        set v_fmt = concat('%m-', v_fmt);
        set v_str = concat(date_format(v_now, '%m'), concat('-', v_str));
    end if;
    if instr(lacks, 'D') > 0 then
        set v_fmt = concat('%d-', v_fmt);
        set v_str = concat(date_format(v_now, '%d'), concat('-', v_str));
    end if;
    return str_to_date(v_str, v_fmt);
END;

-- SQL --  删除unisql_regexp_substr函数
DROP FUNCTION IF EXISTS unisql.`unisql_regexp_substr`;

-- SQL --  创建unisql_regexp_substr函数
CREATE FUNCTION unisql.`unisql_regexp_substr`(input_string VARCHAR(1000), pattern VARCHAR(1000), position INT, occurrence INT)
RETURNS VARCHAR(1000)
DETERMINISTIC
NO SQL
BEGIN
    DECLARE found BOOLEAN;
    DECLARE use_str VARCHAR(1000);
    DECLARE result VARCHAR(1000);
    SET found = FALSE;
    SET use_str = SUBSTRING(input_string, position);
    SET result = NULL;

    SET @x := 0;
    WHILE @x < occurrence DO
        SET @usePattern := pattern;
        IF instr(@usePattern, '^') != 1 THEN
            SET @usePattern := CONCAT('^', @usePattern);
        END IF;

        SET @substr := use_str;
        SET @inputLen := CHAR_LENGTH(use_str);

        SET @i := 1;
        loop2: WHILE @i <= @inputLen DO
            SET @substr := SUBSTRING(@substr, @i);
            IF @substr REGEXP @usePattern THEN
                SET use_str := @substr;
                SET found = TRUE;
                leave loop2;
            END IF;
            SET @i := @i + 1;
        END WHILE;

        IF NOT found THEN
            RETURN NULL;
        END IF;

        SET found = FALSE;

        IF @usePattern NOT REGEXP '\\$$' THEN
            SET @usePattern := CONCAT(@usePattern, '$');
        END IF;
        SET @inputLen := CHAR_LENGTH(@substr);

        SET @i := @inputLen;
        loop3: WHILE @i > 0 DO
            SET @substr := SUBSTRING(@substr, 1, @i);
            IF @substr REGEXP @usePattern THEN
                SET found = TRUE;
                leave loop3;
            END IF;
            SET @i := @i - 1;
        END WHILE;

        IF found THEN
            SET result := @substr;
            SET use_str := SUBSTRING(use_str, CHAR_LENGTH(@substr) + 1);
        ELSE
            RETURN NULL;
        END IF;

        SET @x := @x + 1;
    END WHILE;

    RETURN result;
end ;
